/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Collections.Specialized;
using System.Collections;
using System.Runtime.Serialization;
using Borland.Eco.Persistence;
using Borland.Eco.Exceptions;
using Borland.Eco.DataRepresentation;
using Borland.Eco.Persistence.ORMapping;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;
using Borland.Eco.Interfaces;

namespace Borland.Eco.Persistence
{
	/// <summary>
	/// This class implements a key mapper for autoincremental fields. The actual id values are allocated by the database
	/// </summary>
	
	public abstract class AbstractAutoIncKeyMapper: GenericKeyMapper, IKeyMapper, IAutoIncIdMapper
	{
		protected AbstractAutoIncKeyMapper(): base() {}
		private ISingleColumnAttributemapping m_IntMapper;
		private IAutoIncAttributemapping m_AutoIncMapper;

		public void IdToParameters(ObjectId id, IDataParameter[] parameters)
		{
			System.Object value;
			if (id != null)
				value = id.Key;
			else
				value = null;
			GetKeyMapper(0).ValueToParameter(value, parameters[0]);
		}

		///<exception cref="EcoException">Thrown if <paramref name="timestamp"/> indicates a request for versioned data.</exception>
		public ObjectId IdFromFields(IField[] keyFields, ArrayList keyMappers, int classId, int timeStamp)
		{
			IField f = keyFields[0];
			if (!f.IsNull)
			{
				System.Object value = GetKeyMapper(0).ColumnToValue(f.Value);
				if (value == null)
					return null;
				else if (timeStamp != VersionConst.CurrentVersion)
					throw new EcoException(InterfacesStringRes.sAutoIncNotSupportingVersionedDBs);
				else
					return new DefaultId(value, classId);
			}
			return null;
		}

		public int ColumnCount()
		{
			return 1;
		}

		public void ReserveId()
		{
			// Intentionally left blank
		}

		public ObjectId NewIdPreUpdate(IDatabase db, ObjectId oldId)
		{
			return null;
		}

		public ObjectId NewIdFromAttributes(ObjectContents objContents, ArrayList memberIndexes, IdTranslationList translationList)
		{
			return null;
		}
		public void InitializePSDescriptions(DatabaseDefinition db)
		{
			// intentionally left blank
		}
		public void InitializeDatabase(IDatabase db)
		{
			// intentionally left blank
		}
		public void InitializeDatabaseScript(StringCollection script)
		{
			// intentionally left blank
		}
		public IdAllocationMode GetIdAllocationMode()
		{
			return IdAllocationMode.PostUpdate;
		}
		public string ColumnType(ArrayList KeyMappers, int columnNum)
		{
			return m_IntMapper.ColumnType(-1);
		}

		public string ColumnTypeForAutoInc()
		{
			return m_AutoIncMapper.ColumnType(-1);
		}

		///<exception cref="InvalidOperationException">Thrown if the MapperDictionary does not contain any implementor of <see cref="IAutoIncAttributemapping"/>.</exception>
		public void Initialize(SqlDatabaseConfig config, ArrayList keyMappers)
		{
			InternalInitialize(config, keyMappers);
			m_IntMapper = MapperDictionary.FindSingleColumnMappingByName(IntAttributeMapperName);
			m_AutoIncMapper = MapperDictionary.FindSingleColumnMappingByName(AutoIncAttributeMapperName) as IAutoIncAttributemapping;

			if (m_AutoIncMapper == null)
				throw new InvalidOperationException(InterfacesStringRes.sUnableToFindImplementor);
		}

		public string IdListToWhereFragment(ObjectIdCollection idList, IParameterized parameterized, StringCollection idColumns)
		{
			return InternalIdListToWhereFragment(idList, parameterized, idColumns, m_IntMapper);
		}

		public string IdToWhereFragment(ObjectId id, IParameterized parameterized, StringCollection idColumns)
		{
			ObjectIdCollection list = new ObjectIdCollection();
			list.Add(id);
			return IdListToWhereFragment(list, parameterized, idColumns);
		}

		public void InitializeAutoIncField(string tableName, string columnName, IDatabase database)
		{
			m_AutoIncMapper.MakeColumnAutoInc(tableName, columnName, database);
		}
		
		protected abstract string AutoIncAttributeMapperName { get; }
		protected abstract string IntAttributeMapperName { get; }
	}
	
	public class AutoIncKeyMapper: AbstractAutoIncKeyMapper, IKeyMapper, IAutoIncIdMapper
	{
		protected override string AutoIncAttributeMapperName { get { return "AutoInc"; } } // do not localize
		protected override string IntAttributeMapperName { get { return "System.Int32"; } } // do not localize
		
	}

	public class AutoInc16KeyMapper: AbstractAutoIncKeyMapper, IKeyMapper, IAutoIncIdMapper
	{
		protected override string AutoIncAttributeMapperName { get { return "AutoInc16"; } } // do not localize
		protected override string IntAttributeMapperName { get { return "System.Int16"; } } // do not localize
	}
}


